/* backup.c: part of the spbackup - programm
 *
 * written by: steffen pohle <steffen@gulpe.de>
 * date: 2013-07-30
 *
 */

#include "backup.h"

int cmd_setup = 0;
int debug = 0;

char *dirsrc = NULL;
char *dirdst = NULL;

unsigned long long int dsize, ssize;
unsigned int dfiles, sfiles;

void help ();
int callback_comparesrc (int res, char *dst, struct stat *dbuf, char *src, struct stat *sbuf);
int callback_comparedst (int res, char *dst, struct stat *dbuf, char *src, struct stat *sbuf);

int fix_mode (char *dstfn, struct stat *sbuf);
int fix_owner (char *dstfn, struct stat *sbuf);
int fix_date (char *fn, struct stat *sbuf);
int copy_file (char *dstfn, char *srcfn, struct stat *sbuf);
int delete_file (char *dstfn);
int create_directory (char *dstfn, mode_t mode);
int delete_directory (char *dstfn);
#define delete_link(_dstfn) delete_file(_dstfn)
int copy_link (char *dstfn, char *srcfn);

int main (int argc, char **argv) {
	int i;

	printf ("spbackup version: %s\n", VERSION);

	for (i = 0; i < argc; i++) {
		if ((strcmp ("-h", argv[i]) == 0) ||
		    (strcmp ("-help", argv[i]) == 0)) {
			help ();
			return 1;
		}
		else if (strcmp ("-s", argv[i]) == 0) {
			i++;
			dirsrc = argv[i];
		}
		else if (strcmp ("-d", argv[i]) == 0) {
			i++;
			dirdst = argv[i];
		}
		else if (strcmp ("-im", argv[i]) == 0) {
			cmd_setup |= CMD_IGNORE_MODE;
		}
		else if (strcmp ("-io", argv[i]) == 0) {
			cmd_setup |= CMD_IGNORE_OWNER;
		}
		else if (strcmp ("-C", argv[i]) == 0) {
			cmd_setup |= CMD_COPY;
		}
		else if (strcmp ("-D", argv[i]) == 0) {
			cmd_setup |= CMD_REMOVE;
		}
		else if (strcmp ("-fm", argv[i]) == 0) {
			cmd_setup |= CMD_FIXMODE;
		}
		else if (strcmp ("-fd", argv[i]) == 0) {
			cmd_setup |= CMD_FIXDATE;
		}
		else if (strcmp ("-nc", argv[i]) == 0) {
			debug |= DBG_NOCHECK;
		}
		else if (strcmp ("-fo", argv[i]) == 0) {
			cmd_setup |= CMD_FIXOWNER;
		}
	}

	if (argc == 1) {
		help ();
		return 1;
	}

	if (dirsrc == NULL || dirdst == NULL) {
		printf ("ERROR: please specify source and destination\n");
		return -1;
	}

	dsize = ssize = 0;
	dfiles = sfiles = 0;

	printf ("\n\nreading src : '%s'\n", dirsrc);
	compdir (0, dirsrc, dirdst, NULL, callback_comparesrc);
	printf ("\n\nreading dst : '%s'\n", dirdst);
	compdir (1, dirdst, dirsrc, NULL, callback_comparedst);

	printf ("\n\n");
	printf ("sfiles: %9d\n", sfiles);
	printf (" ssize: %9lld MBytes\n", ssize/1024/1024);
	printf ("dfiles: %9d\n", dfiles);
	printf (" dsize: %9lld MBytes\n", dsize/1024/1024);

	return 0;
};


void help () {
	printf ("spbackup: comparing two directorys but to save some\n");
	printf ("          time, there will be no compare of the file\n");
	printf ("          itself.  (only time and size are checked)\n");
	printf ("          If there are differences, the files can be \n");
	printf ("          copied to the destination. If the files do\n");
	printf ("          not exsist on the source they can be deleted\n");
	printf ("          from the source.\n");
	printf ("\n");
	printf ("    -h            displaying this help.\n");
	printf ("    -s path       source path\n");
	printf ("    -d path       destination path\n");
	printf ("    -im           ignore permission mode\n");
	printf ("    -io           ignore owner and user\n");
	printf ("    -C            copy missing or changed files from the\n");
	printf ("                  source to the destination.\n");
	printf ("    -D            delete files on the destination if they\n");
	printf ("                  do not anymore exists on the source.\n");
	printf ("    -fo           fix owner and group ids on destination.\n");
	printf ("    -fm           fix permission modes on destination.\n");
	printf ("    -fd           fix date on destination.\n");
	printf ("    -nc           display no check message.\n");
	printf ("\n");
	printf ("    written by steffen pohle <steffen(AT)gulpe.de> @ 2013\n");
	printf ("    the programm is free to use and free to any change.\n");
	printf ("    no warrenty and no cover for any damage this program \n");
	printf ("    might do is given.    USE IT AT YOUR OWN RISK.\n\n");
};


int callback_comparesrc (int res, char *dst, struct stat *dbuf, char *src, struct stat *sbuf) {
	/* fix mode if needed */
	if (!(cmd_setup & CMD_IGNORE_MODE) && (res & CDR_DIFFMODE)) {
		if ((res & CDR_ISDIR) || (res & CDR_ISFILE)) {
			if (cmd_setup & CMD_FIXMODE) {
				printf ("diffmode file/link %s\n", src);
				fix_mode (dst, sbuf);
			}
			else printf ("diffmode file/link %s\n", src);
		}
	}

	/* fix owner if needed */
	if (!(cmd_setup & CMD_IGNORE_OWNER) && (res & CDR_DIFFOWNER)) {
		if ((res & CDR_ISDIR) || (res & CDR_ISFILE)) {
			if (cmd_setup & CMD_FIXOWNER) {
				printf ("diffuser file/link %s\n", src);
				fix_owner (dst, sbuf);
			}
			else printf ("diffuser file/link %s\n", src);
		}
	}

	/* only found on source */
	if (res & CDR_NODEST) {
		if (cmd_setup & CMD_COPY) {
			if (res & CDR_ISLINK) {
				printf ("copy link %s\n", src);
				copy_link (dst, src);
			}
			if (res & CDR_ISDIR) {
				printf ("create dir %s\n", src);
				create_directory (dst, sbuf->st_mode);
				if (cmd_setup & CMD_FIXMODE)  fix_mode (dst, sbuf);
				if (cmd_setup & CMD_FIXOWNER) fix_owner (dst, sbuf);
			}
			if (res & CDR_ISFILE) {
				printf ("copy file %s\n", src);
				copy_file (dst, src, sbuf);
				if (cmd_setup & CMD_FIXMODE)  fix_mode (dst, sbuf);
				if (cmd_setup & CMD_FIXOWNER) fix_owner (dst, sbuf);
			}
		}
		else
			printf ("nodest %s\n", src);
	}

	if (res & CDR_ISLINK) {
		if ((res & CDR_ISDIFF) == res) {
			if (cmd_setup & CMD_COPY) {
				printf ("different link %s\n", src);
				delete_link (dst);
				copy_link (dst, src);
			}
			else printf ("different link %s\n", src);
		}
	}

	if (res & CDR_ISFILE) {
		if (res & CDR_DIFFSIZE) {
			if (cmd_setup & CMD_COPY) {
				printf ("diffsize_copy file %s\n", src);
				delete_file (dst);
				copy_file (dst, src, sbuf);
				if (cmd_setup & CMD_FIXMODE)  fix_mode (dst, sbuf);
				if (cmd_setup & CMD_FIXOWNER) fix_owner (dst, sbuf);
			}
			else printf ("diffsize file %s\n", src);
		}

		else if (res & CDR_DIFFDATE) {
			if (cmd_setup & CMD_FIXDATE) {
				printf ("diffdate file %s\n", src);
				fix_date(dst, sbuf);
			}
			else if (cmd_setup & CMD_COPY) {
				printf ("diffdate_copy file %s\n", src);
				delete_file (dst);
				copy_file (dst, src, sbuf);
				if (cmd_setup & CMD_FIXMODE)  fix_mode (dst, sbuf);
				if (cmd_setup & CMD_FIXOWNER) fix_owner (dst, sbuf);
			}
			else printf ("diffdate file %s\n", src);
		}
	}

	if (res & CDR_ISFILE) {
		sfiles++;
		ssize += sbuf->st_size;
	}

	return CDCB_RECURSIVE;
};


int callback_comparedst (int res, char *dst, struct stat *dbuf, char *src, struct stat *sbuf) {
	if (res & CDR_NODEST) {
		if ((cmd_setup & CMD_REMOVE) && (res & CDR_ISFILE)) {
			printf ("onlydest file %s\n", src);
			delete_file (src);
		}
		else if ((cmd_setup & CMD_REMOVE) && (res & CDR_ISLINK)) {
			printf ("onlydest link %s\n", src);
			delete_link (src);
		}
		else if ((cmd_setup & CMD_REMOVE) && (res & CDR_ISDIR)) {
			printf ("onlydest dir %s\n", src);
			delete_directory (src);
			return CDCB_NEXT;
		}
		else printf ("onlydest dir %s\n", src);
	}

	if (res & CDR_ISFILE) {
		dfiles++;
		dsize += sbuf->st_size;
	}
	return CDCB_RECURSIVE;
};


int fix_mode (char *dstfn, struct stat *sbuf) {
	int i;

	i = chmod (dstfn, sbuf->st_mode);
	if (i) {
		printf ("fixmode on file '%s' returned error: %s\n", dstfn, strerror (errno));
	}
	return i;
};


int fix_owner (char *dstfn, struct stat *sbuf) {
	int i;

	i = chown (dstfn, sbuf->st_uid, sbuf->st_gid);
	if (i) {
		printf ("fixowner on file '%s' returned error: %s\n", dstfn, strerror (errno));
	}
	return i;
};


#define BUFFER_SIZE 0x10000
int copy_file (char *dstfn, char *srcfn, struct stat *sbuf) {
	char buffer[BUFFER_SIZE];
	int dfd, sfd, i, j, ds = 0, ss = 0;
	struct utimbuf utbuf;

	if ((sfd = open (srcfn, O_RDONLY)) <= 0) {
		printf ("copy_file: open source file '%s' failed: %s\n", srcfn, strerror(errno));
		return 1;
	}

	if ((dfd = creat (dstfn, S_IRUSR | S_IWUSR)) <= 0) {
		printf ("copy_file: creating destiantion file '%s' failed: %s\n", dstfn, strerror(errno));
		close (sfd);
		return 1;
	}

	do {
		i = read (sfd, buffer, BUFFER_SIZE);
		j = write (dfd, buffer, i);
		ds += j;
		ss += i;
	} while (i == BUFFER_SIZE && i == j);

	if (i != j) {
		printf ("copy_file: something went wrong with reading/writing file: %s\n", dstfn);
		printf ("copy_file: size is wrong source:%d  destination:%d\n", i, j);
		i = 1;
	}
	else i = 0;
	close (sfd);
	close (dfd);

	utbuf.actime = sbuf->st_atime;
	utbuf.modtime = sbuf->st_mtime;

	if ((i = utime(dstfn, &utbuf))) {
		printf ("copy_file: error in setting up the time on file: %s: error:%s\n", dstfn, strerror(errno));
	}

	return i;
};


int fix_date (char *fn, struct stat *sbuf) {
	int i;
	struct utimbuf utbuf;

	utbuf.actime = sbuf->st_atime;
	utbuf.modtime = sbuf->st_mtime;

	if ((i = utime(fn, &utbuf))) {
		printf ("copy_file: error in setting up the time on file: %s: error:%s\n", fn, strerror(errno));
	}

	return i;
}

int delete_file (char *dstfn) {
	int i;

	i = unlink (dstfn);
	if (i) {
		printf ("delete_file '%s' returned error: %s\n", dstfn, strerror (errno));
	}
	return i;
};


int create_directory (char *dstfn, mode_t mode) {
	int i;

	i = mkdir (dstfn, mode);
	if (i) {
		printf ("create direcotry '%s' returned error: %s\n", dstfn, strerror (errno));
	}
	return i;
};


int delete_directory (char *dstfn) {
	DIR *srcdir;
	struct dirent *entry;
	char fname[PATHNAME_LEN];
	char tmpfn[PATHNAME_LEN];

	strncpy (fname, dstfn, PATHNAME_LEN);
	str_delslash (fname);

	srcdir = opendir (fname);
	if (srcdir == NULL) {
		printf ("delete_directory: directory '%s' could not been opened.\n", fname);
		return 0;
	}

	for (entry = readdir (srcdir) ;entry != NULL; entry = readdir (srcdir)) {
		if (strcmp (entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0) continue;
		snprintf (tmpfn, PATHNAME_LEN, "%s/%s", fname, entry->d_name);
		if (entry->d_type == DT_DIR) {
			if (delete_directory (tmpfn)) {
				printf ("delete_directory: could not delete: %s\n", tmpfn);
			}
		}
		else {
			if (unlink (tmpfn)) {
				printf ("delete_directory: could not delete file/link: %s Error:%s\n", tmpfn, strerror(errno));
			}
		}
	}

	closedir (srcdir);
	if (rmdir (fname)) {
		printf ("delete_directory: deleting %s Error:%s\n", fname, strerror(errno));
		return 1;
	} else return 0;
};


int copy_link (char *dstfn, char *srcfn) {
	char slnk[PATHNAME_LEN];
	int len, i;

	len = readlink(srcfn, slnk, PATHNAME_LEN);
	if (len < 0) {
		printf ("copy link: error reading link '%s' %s\n", srcfn, strerror (errno));
		slnk[0] = 0;
		return 0;
	}
	else slnk[len] = 0;

	i = symlink (slnk, dstfn);
	if (i) {
		printf ("copy link: error creating link: '%s' Error:%s\n.", dstfn, strerror(errno));
	}

	printf ("   finished.\n");
	return i;
};


